<?php

namespace ZS\Usaepay\Helper;

use ZS\Usaepay\Model\CardFactory;

/**
 * Class Card
 * @package ZS\Usaepay\Helper
 */
class Card
{
    /**
     * @var CardFactory
     */
    protected $cardFactory;

    /**
     * Card constructor.
     * @param CardFactory $cardFactory
     */
    public function __construct(
        CardFactory $cardFactory
    )
    {
        $this->cardFactory = $cardFactory;
    }

    /**
     * @param \ZS\Usaepay\Model\Card $card
     * @return string
     */
    public function getCardLabel(\ZS\Usaepay\Model\Card $card)
    {
        $expMonth = $this->getExpMonth($card);
        $expYear = $this->getExpYear($card);
        $ccType = $this->getExtendedCcType($card->getData('cc_type'));
        return "$ccType {$this->getLast4($card)} - Expires: {$expMonth}/{$expYear}";
    }

    /**
     * @param \ZS\Usaepay\Model\Card $card
     * @return array
     */
    public function getScrubbedCardData(\ZS\Usaepay\Model\Card $card)
    {
        return [
            'exp_month' => $this->getExpMonth($card),
            'exp_year'=> $this->getExpYear($card),
            'card_number' => str_repeat('*', 12) . $card->getData('cc_last_4'),
            'card_type' => $card->getData('cc_type')
        ];
    }

    /**
     * @param \ZS\Usaepay\Model\Card $card
     * @return int
     */
    public function getExpMonth(\ZS\Usaepay\Model\Card $card)
    {
        $expData = $card->getData('cc_exp_month');
        return str_pad($expData, 2, '0', STR_PAD_LEFT);
    }

    /**
     * @param \ZS\Usaepay\Model\Card $card
     * @return int
     */
    public function getExpYear(\ZS\Usaepay\Model\Card $card)
    {
        return $card->getData('cc_exp_year');
    }

    /**
     * @param \ZS\Usaepay\Model\Card $card
     * @return string
     */
    public function getLast4(\ZS\Usaepay\Model\Card $card)
    {
        return '****-' . $card->getData('cc_last_4');
    }

    /**
     * @param string $cardType
     * @return string
     */
    private function getExtendedCcType($cardType)
    {
        switch ($cardType) {
            case 'VI':
                return 'Visa';
            case 'MC':
                return 'MasterCard';
            case 'AE':
                return 'American Express';
            case 'DC':
                return 'Diners Club';
            case 'DI':
                return 'Discover';
        }
        return $cardType;
    }

    /**
     * @param $customerId
     * @return array
     */
    public function getSavedCardsByCustomer($customerId)
    {
        $cards = $this->cardFactory->create()->getCollection();
        $cards->addFieldToFilter('customer_id', $customerId);

        $result = [];

        foreach ($cards as $card) {
            $result[] = [
                'extendedData' => $this->getScrubbedCardData($card),
                'cardId' => $card->getId(),
                'label' => $this->getCardLabel($card)
            ];
        }

        return $result;
    }

}